import math
import matplotlib.pyplot as plt

# Funzione per calcolare le emissioni per ogni fase
def calculate_emissions(gCO2tot, num_passengers, num_kilometers):
    # Percentuali per ogni fase
    manufacturing_percentage = 3.52 / 100
    operations_percentage = 78.63 / 100
    fuel_production_percentage = 2.07 / 100
    infrastructure_operation_percentage = 3.5 / 100
    infrastructure_construction_percentage = 12.29 / 100

    # Calcolo delle emissioni per ogni fase
    manufacturing = manufacturing_percentage * gCO2tot * num_passengers * num_kilometers
    operations = operations_percentage * gCO2tot * num_passengers * num_kilometers
    fuel_production = fuel_production_percentage * gCO2tot * num_passengers * num_kilometers
    infrastructure_operation = infrastructure_operation_percentage * gCO2tot * num_passengers * num_kilometers
    infrastructure_construction = infrastructure_construction_percentage * gCO2tot * num_passengers * num_kilometers

    # Somma totale delle emissioni
    total_emissions = manufacturing + operations + fuel_production + infrastructure_operation + infrastructure_construction

    return manufacturing, operations, fuel_production, infrastructure_operation, infrastructure_construction, total_emissions

# Funzione per calcolare e moltiplicare i valori in base al parametro D
def calculate_values_and_multiply(num_kilometers, total_emissions):
    parameter_D = num_kilometers / 1000

    CO2 = 1.0
    NOx = 2.3 * math.atan(3.1 * parameter_D) - 2.0
    CiC = 1.1 * math.atan(0.5 * parameter_D)
    H2O = 0.2 * math.atan(parameter_D)

    CO2_multiplied = CO2 * total_emissions
    NOx_multiplied = NOx * total_emissions
    CiC_multiplied = CiC * total_emissions
    H2O_multiplied = H2O * total_emissions

    # Somma di tutti i valori moltiplicati
    total_sum = CO2_multiplied + NOx_multiplied + CiC_multiplied + H2O_multiplied

    return CO2_multiplied, NOx_multiplied, CiC_multiplied, H2O_multiplied, total_sum

# Funzione per calcolare i pesi dei componenti in base alle percentuali del MLW
def calculate_weights(mlw):
    wing_weight = 0.27 * mlw
    fuselage_weight = 0.19 * mlw
    engine_weight = 0.20 * mlw
    h_stabilizer_weight = 0.11 * mlw
    v_stabilizer_weight = 0.06 * mlw
    landing_gear_weight = 0.04 * mlw
    other_weight = 0.13 * mlw

    return wing_weight, fuselage_weight, engine_weight, h_stabilizer_weight, v_stabilizer_weight, other_weight, landing_gear_weight

# Funzione per calcolare i pesi di ogni materiale
def calculate_component_weights(mlw):
    aluminum_percentage = 0.68
    composites_percentage = 0.15
    steel_percentage = 0.09
    titanium_percentage = 0.06
    miscellaneous_percentage = 0.02

    aluminum_weight = aluminum_percentage * mlw
    composites_weight = composites_percentage * mlw
    steel_weight = steel_percentage * mlw
    titanium_weight = titanium_percentage * mlw
    miscellaneous_weight = miscellaneous_percentage * mlw

    return aluminum_weight, composites_weight, steel_weight, titanium_weight, miscellaneous_weight

# Funzione per calcolare le emissioni dei materiali
def calculate_emissions_for_materials(aluminum_weight, composites_weight, steel_weight, titanium_weight, miscellaneous_weight):
    # Fattori di emissione per riciclo, incenerimento, e smaltimento in discarica
    aluminum_recycling_emissions = 2  # kgCO2/kg
    composites_incineration_emissions = 3  # kgCO2/kg
    steel_recycling_emissions = 1.5  # kgCO2/kg
    titanium_recycling_emissions = 4  # kgCO2/kg
    miscellaneous_landfill_emissions = 5  # kgCO2/kg
    miscellaneous_incineration_emissions = 1  # kgCO2/kg

    # Percentuali di smaltimento
    recycling_percentage = 100 / 100
    landfill_percentage = 50 / 100
    incineration_percentage = 50 / 100

    # Calcolo delle emissioni
    aluminum_total_emissions = aluminum_weight * aluminum_recycling_emissions * recycling_percentage
    composites_total_emissions = composites_weight * composites_incineration_emissions * incineration_percentage
    steel_total_emissions = steel_weight * steel_recycling_emissions * recycling_percentage
    titanium_total_emissions = titanium_weight * titanium_recycling_emissions * recycling_percentage
    miscellaneous_total_emissions = (miscellaneous_weight * miscellaneous_landfill_emissions * landfill_percentage) + \
                                    (miscellaneous_weight * miscellaneous_incineration_emissions * incineration_percentage)

    total_emissions_EoL = aluminum_total_emissions + composites_total_emissions + steel_total_emissions + titanium_total_emissions + miscellaneous_total_emissions

    return total_emissions_EoL, aluminum_total_emissions, composites_total_emissions, steel_total_emissions, titanium_total_emissions, miscellaneous_total_emissions

def main():
    # Input per il calcolo delle emissioni
    gCO2tot = float(input("Enter the number of grams of CO2 emitted (gCO2tot): "))
    num_passengers = int(input("Enter the number of passengers: "))
    num_kilometers = float(input("Enter the number of flight kilometers: "))

    # Calcolo delle emissioni totali per fase
    manufacturing, operations, fuel_production, infrastructure_operation, infrastructure_construction, total_emissions = calculate_emissions(gCO2tot, num_passengers, num_kilometers)

    # Calcolo dei valori e moltiplicazione per le emissioni totali
    CO2_multiplied, NOx_multiplied, CiC_multiplied, H2O_multiplied, total_sum = calculate_values_and_multiply(num_kilometers, total_emissions)

    # Input per i pesi dei componenti e calcolo delle emissioni di fine vita (EoL)
    mlw = float(input("Enter the maximum landing weight (MLW) in kg: "))

    # Calcolo del peso di ogni componente
    wing_weight, fuselage_weight, engine_weight, h_stabilizer_weight, v_stabilizer_weight, other_weight, landing_gear_weight = calculate_weights(mlw)

    # Calcolo del peso di ogni materiale
    aluminum_weight, composites_weight, steel_weight, titanium_weight, miscellaneous_weight = calculate_component_weights(mlw)

    # Calcolo delle emissioni per ogni materiale
    total_emissions_EoL, aluminum_emissions, composites_emissions, steel_emissions, titanium_emissions, miscellaneous_emissions = calculate_emissions_for_materials(aluminum_weight, composites_weight, steel_weight, titanium_weight, miscellaneous_weight)

    # Conversione delle emissioni di fine vita da kg a g e calcolo delle emissioni totali dell'aereo
    total_emissions_EoL_g = total_emissions_EoL * 1000
    total_aircraft_emissions = total_sum + total_emissions_EoL_g

    # Calcolo del contributo delle emissioni di ciascuna struttura
    contribution_landing_gear = (landing_gear_weight / mlw) * total_aircraft_emissions
    contribution_engine = (engine_weight / mlw) * total_aircraft_emissions
    contribution_wing = (wing_weight / mlw) * total_aircraft_emissions
    contribution_h_stabilizer = (h_stabilizer_weight / mlw) * total_aircraft_emissions
    contribution_v_stabilizer = (v_stabilizer_weight / mlw) * total_aircraft_emissions
    contribution_other = (other_weight / mlw) * total_aircraft_emissions
    contribution_fuselage = (fuselage_weight / mlw) * total_aircraft_emissions

    # Stampa dei risultati
    print("\nCalculated material weights:")
    print(f"Aluminum weight: {aluminum_weight:.2f} kg")
    print(f"Composites weight: {composites_weight:.2f} kg")
    print(f"Steel weight: {steel_weight:.2f} kg")
    print(f"Titanium weight: {titanium_weight:.2f} kg")
    print(f"Miscellaneous weight: {miscellaneous_weight:.2f} kg")

    print("\nEmissions for each material:")
    print(f"Aluminum emissions: {aluminum_emissions:.2f} kgCO2")
    print(f"Composites emissions: {composites_emissions:.2f} kgCO2")
    print(f"Steel emissions: {steel_emissions:.2f} kgCO2")
    print(f"Titanium emissions: {titanium_emissions:.2f} kgCO2")
    print(f"Miscellaneous emissions: {miscellaneous_emissions:.2f} kgCO2")

    print("\nEmissions from End of Life (EoL):")
    print(f"Total EoL emissions: {total_emissions_EoL:.2f} kgCO2")
    print(f"Total aircraft emissions including EoL: {total_aircraft_emissions:.2f} gCO2")

    print("\nTotal Aircraft Emissions (excluding EoL):")
    print(f"{total_sum:.2f} gCO2")

    print("\nEmissions from each phase:")
    print(f"Manufacturing: {manufacturing:.2f} gCO2")
    print(f"Operations: {operations:.2f} gCO2")
    print(f"Fuel Production: {fuel_production:.2f} gCO2")
    print(f"Infrastructure Operation: {infrastructure_operation:.2f} gCO2")
    print(f"Infrastructure Construction: {infrastructure_construction:.2f} gCO2")
    print(f"Total emissions from phases: {total_emissions:.2f} gCO2")

    print("\nThe multiplied values are:")
    print(f"CO2 = {CO2_multiplied:.2f} gCO2")
    print(f"NOx = {NOx_multiplied:.2f} gCO2")
    print(f"CiC = {CiC_multiplied:.2f} gCO2")
    print(f"H2O = {H2O_multiplied:.2f} gCO2")
    print(f"total_sum = {total_sum:.2f} gCO2")

    print("\nContribution of each component to total emissions:")
    print(f"Landing Gear: {contribution_landing_gear:.2f} gCO2")
    print(f"Engine: {contribution_engine:.2f} gCO2")
    print(f"Wing: {contribution_wing:.2f} gCO2")
    print(f"Horizontal Stabilizer: {contribution_h_stabilizer:.2f} gCO2")
    print(f"Vertical Stabilizer: {contribution_v_stabilizer:.2f} gCO2")
    print(f"Other Components: {contribution_other:.2f} gCO2")
    print(f"Fuselage: {contribution_fuselage:.2f} gCO2")

    # Grafico a dispersione per le emissioni di CO2 per ogni struttura
    def plot_scatter(components, component_emissions):
        fig, ax = plt.subplots(figsize=(10, 6))

        colors = ['b', 'g', 'r', 'c', 'm', 'y', 'k']
        for i, (component, emission) in enumerate(zip(components, component_emissions)):
            ax.scatter(i, emission, s=100, color=colors[i], label=component)

        ax.set_xlabel('Components')
        ax.set_ylabel('CO2 Emissions (gCO2)')
        ax.set_title('CO2 Emissions per Structure Component')
        ax.legend()
        plt.xticks(range(len(components)), components, rotation=45)
        plt.grid(True)
        plt.tight_layout()
        plt.show()

    # Chiamata alla funzione di plot per scatter
    components = ['Wing', 'Fuselage', 'Engine', 'Horizontal Stabilizer', 'Vertical Stabilizer', 'Other', 'Landing Gear']
    component_emissions = [contribution_wing, contribution_fuselage, contribution_engine, contribution_h_stabilizer, contribution_v_stabilizer, contribution_other, contribution_landing_gear]

    plot_scatter(components, component_emissions)

    # Grafico a bolle con legenda per le strutture
    def plot_bubble(components, component_emissions):
        fig, ax = plt.subplots(figsize=(10, 6))

        colors = ['b', 'g', 'r', 'c', 'm', 'y', 'k']
        sizes = [emission * 10 for emission in component_emissions]  # Dimensione delle bolle proporzionale alle emissioni

        for i, (component, emission) in enumerate(zip(components, component_emissions)):
            ax.scatter(i, emission, s=sizes[i], color=colors[i], alpha=0.5, label=f'{component}: {emission:.2f} kgCO2')

        ax.set_xlabel('Components')
        ax.set_ylabel('CO2 Emissions (gCO2)')
        ax.set_title('CO2 Emissions per Structure Component')
        ax.legend()
        plt.xticks(range(len(components)), components, rotation=45)
        plt.grid(True)
        plt.tight_layout()
        plt.show()

    # Chiamata alla funzione di plot per bubble
    plot_bubble(components, component_emissions)

    # Grafico a torta per mostrare il contributo delle emissioni dei materiali
    def plot_pie(material_labels, material_emissions):
        fig, ax = plt.subplots(figsize=(8, 8))
        ax.pie(material_emissions, labels=[f'{label}: {emission:.2f} kgCO2' for label, emission in zip(material_labels, material_emissions)], autopct='%1.1f%%', startangle=140)
        ax.set_title('Material Emissions Contributions')
        plt.tight_layout()
        plt.show()

    # Chiamata alla funzione di plot per pie
    material_labels = ['Aluminum', 'Composites', 'Steel', 'Titanium', 'Miscellaneous']
    material_emissions = [aluminum_emissions, composites_emissions, steel_emissions, titanium_emissions, miscellaneous_emissions]

    plot_pie(material_labels, material_emissions)

    # Grafico a barre per le emissioni di CO2 per fase
    def plot_bar_phases(manufacturing, operations, fuel_production, infrastructure_operation, infrastructure_construction):
        phases = ['Manufacturing', 'Operations', 'Fuel Production', 'Infrastructure Operation', 'Infrastructure Construction']
        emissions = [manufacturing, operations, fuel_production, infrastructure_operation, infrastructure_construction]

        fig, ax = plt.subplots(figsize=(10, 6))
        ax.bar(phases, emissions, color=['b', 'g', 'r', 'c', 'm'])
        ax.set_xlabel('Phases')
        ax.set_ylabel('CO2 Emissions (gCO2)')
        ax.set_title('CO2 Emissions per Phase')
        plt.xticks(rotation=45)
        plt.tight_layout()
        plt.show()

    # Chiamata alla funzione di plot per bar
    plot_bar_phases(manufacturing, operations, fuel_production, infrastructure_operation, infrastructure_construction)

    # Grafico a barre per le emissioni di CO2, NOx, CiC e H2O
    def plot_emissions_by_pollutant(pollutants, pollutant_emissions):
        plt.figure(figsize=(10, 6))
        plt.bar(pollutants, pollutant_emissions, color='green')
        plt.xlabel('Pollutants')
        plt.ylabel('Emissions (gCO2)')
        plt.title('Emissions by Pollutant')
        plt.show()

    # Chiamata alla funzione di plot per le emissioni per inquinante
    pollutants = ['CO2', 'NOx', 'CiC', 'H2O']
    pollutant_emissions = [CO2_multiplied, NOx_multiplied, CiC_multiplied, H2O_multiplied]

    plot_emissions_by_pollutant(pollutants, pollutant_emissions)

if __name__ == "__main__":
    main()
